#ifndef PECK_FFT_H
#define PECK_FFT_H

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifdef USE_SIMD
# include <xmmintrin.h>
# define peck_fft_scalar __m128
#define PECK_FFT_MALLOC(nbytes) _mm_malloc(nbytes, 16)
#define PECK_FFT_FREE _mm_free
#else	
#define PECK_FFT_MALLOC malloc
#define PECK_FFT_FREE free
#endif	


#ifdef FIXED_POINT
#include <sys/types.h>	
# if (FIXED_POINT == 32)
#  define peck_fft_scalar int32_t
# else	
#  define peck_fft_scalar int16_t
# endif
#else
# ifndef peck_fft_scalar
/*  default is float */
#   define peck_fft_scalar float
# endif
#endif

typedef struct {
    peck_fft_scalar r;
    peck_fft_scalar i;
} peck_fft_cpx;

typedef struct peck_fft_state* peck_fft_cfg;

/* 
 *  peck_fft_alloc()
 *  
 *  Initialize a FFT (or IFFT) algorithm's cfg buffer.
 *
 *  typical usage: peck_fft_cfg mycfg = peck_fft_alloc(1024, 0, NULL, NULL);
 *
 *  The return value from fft_alloc() is a cfg buffer used internally
 *  by the FFT routine or NULL.
 *
 *  If lenmem is NULL, then peck_fft_alloc() will allocate a cfg buffer using malloc.
 *  The returned value should be free()d when done to avoid memory leaks.
 *  
 *  The state can be placed in a user supplied buffer 'mem':
 *  If lenmem is not NULL and mem is not NULL and *lenmem is large enough,
 *      then the function places the cfg in mem and the size used in *lenmem
 *      and returns mem.
 *  
 *  If lenmem is not NULL and ( mem is NULL or *lenmem is not large enough),
 *      then the function returns NULL and places the minimum cfg 
 *      buffer size in *lenmem.
 */

peck_fft_cfg peck_fft_alloc(int nfft,int inverse_fft,void * mem,size_t * lenmem); 

/*
 * peck_fft(cfg, in_buf, out_buf)
 *
 * Perform an FFT on a complex input buffer.
 * For a forward FFT,
 *   fin should be  f[0], f[1], ... , f[nfft-1]
 *   fout will be   F[0], F[1], ... , F[nfft-1]
 * Note that each element is complex and can be accessed like
 *   f[k].r and f[k].i
 */
void peck_fft(peck_fft_cfg cfg, const peck_fft_cpx *fin, peck_fft_cpx *fout);

/* If peck_fft_alloc allocated a buffer, it is one contiguous 
 * buffer and can be simply free()d when no longer needed
 */
#define peck_fft_free free

/*
 * Cleans up some memory that gets managed internally. Not necessary to call, but it might clean up 
 * your compiler output to call this before you exit.
 */
void peck_fft_cleanup(void);

/*
 * Returns the smallest integer k, such that k>=n and k has only 'fast' factors (2,3,5)
 */
int peck_fft_next_fast_size(int n);

/* for real ffts, we need an even size */
#define peck_fftr_next_fast_size_real(n) \
    (peck_fft_next_fast_size(((n)+1) >> 1) << 1)

#ifdef __cplusplus
} 
#endif

#endif
