/*
Copyright (c) 2003-2010, Mark Borgerding

All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the author nor the names of any contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/* peck_fft.h
 * defines peck_fft_scalar as either short or a float type and defines
 * typedef struct { peck_fft_scalar r; peck_fft_scalar i; } peck_fft_cpx; 
 */
#include "peck_fft.h"
#include <limits.h>

#define MAXFACTORS 32
/* e.g. an fft of length 128 has 4 factors 
 * as far as kissfft is concerned: 4*4*4*2
 */

struct peck_fft_state{
    int nfft;
    int inverse;
    int factors[2*MAXFACTORS];
    peck_fft_cpx twiddles[1];
};

/*
 * Explanation of macros dealing with complex math:
 *   C_MUL(m,a,b)         : m = a*b
 *   C_SUB( res, a,b)     : res = a - b
 *   C_SUBFROM( res , a)  : res -= a
 *   C_ADDTO( res , a)    : res += a
 */

#define S_MUL(a, b) ((a) * (b))
#define C_MUL(m, a, b) \
    do { \
        (m).r = (a).r*(b).r - (a).i*(b).i; \
        (m).i = (a).r*(b).i + (a).i*(b).r; \
    } while(0)
#define C_MULBYSCALAR(c, s ) \
    do { \
        (c).r *= (s); \
        (c).i *= (s); \
    } while(0)
#define  C_ADD(res, a, b) \
    do { \
	    (res).r = (a).r + (b).r; \
	    (res).i = (a).i + (b).i; \
    } while(0)
#define  C_SUB(res, a, b) \
    do { \
	    (res).r = (a).r - (b).r; \
	    (res).i = (a).i - (b).i; \
    } while(0)
#define C_ADDTO(res, a) \
    do { \
	    (res).r += (a).r; \
	    (res).i += (a).i; \
    } while(0)
#define C_SUBFROM(res, a) \
    do { \
	    (res).r -= (a).r; \
	    (res).i -= (a).i; \
    } while(0)

#if USE_SIMD == SIMD_SSE2
    #define PECK_FFT_COS(phase) _mm_set1_ps(cosf(phase))
    #define PECK_FFT_SIN(phase) _mm_set1_ps(sinf(phase))
    #define HALF_OF(x) ((x)*_mm_set1_ps(0.5f))
#elif USE_SIMD == SIMD_NEON4
    #define PECK_FFT_COS(phase) vdupq_n_f32(cosf(phase))
    #define PECK_FFT_SIN(phase) vdupq_n_f32(sinf(phase))
    #define HALF_OF(x) ((x)*vdupq_n_f32(0.5f))
#elif USE_SIMD == SIMD_NEON2
    #define PECK_FFT_COS(phase) vdup_n_f32(cosf(phase))
    #define PECK_FFT_SIN(phase) vdup_n_f32(sinf(phase))
    #define HALF_OF(x) ((x)*vdup_n_f32(0.5f))
#else
    #define PECK_FFT_COS(phase) (peck_fft_scalar) cosf(phase)
    #define PECK_FFT_SIN(phase) (peck_fft_scalar) sinf(phase)
    #define HALF_OF(x) ((x)*0.5f)
#endif

#define  kf_cexp(x,phase) \
	do { \
		(x)->r = PECK_FFT_COS(phase); \
		(x)->i = PECK_FFT_SIN(phase); \
	} while(0)

#ifdef PECK_FFT_USE_ALLOCA
// define this to allow use of alloca instead of malloc for temporary buffers
// Temporary buffers are used in two case: 
// 1. FFT sizes that have "bad" factors. i.e. not 2,3 and 5
// 2. "in-place" FFTs.  Notice the quotes, since kissfft does not really do an in-place transform.
#include <alloca.h>
#define  PECK_FFT_TMP_ALLOC(nbytes) alloca(nbytes)
#define  PECK_FFT_TMP_FREE(ptr) 
#else
#define  PECK_FFT_TMP_ALLOC(nbytes) PECK_FFT_MALLOC(nbytes)
#define  PECK_FFT_TMP_FREE(ptr) PECK_FFT_FREE(ptr)
#endif
