/*
Copyright (c) 2003-2010, Mark Borgerding

All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the author nor the names of any contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "armv7_cycles.h"
#include "_peck_fft_guts.h"
/* The guts header contains all the multiplication and addition macros that are defined for
 * fixed or floating point complex numbers. It also delares the kf_ internal functions.
 */
#if !BFLY2_ASM
static void kf_bfly2(
    peck_fft_cpx * __restrict Fout,
    const size_t fstride,
    const peck_fft_cfg st,
    int m) {

// printf("kf_bfly2, %d\n", fstride);

    peck_fft_cpx * __restrict tw1 = st->twiddles;
    peck_fft_cpx * __restrict Fout2 = Fout + m;
    do {
        peck_fft_cpx t;
        C_MUL(t, *Fout2, *tw1);
        tw1 += fstride;
        C_SUB(*Fout2, *Fout, t);
        C_ADDTO(*Fout, t);
        ++Fout2;
        ++Fout;
    } while (--m);
}
#endif

#if !BFLY4_ASM
static void kf_bfly4(
    peck_fft_cpx * __restrict Fout,
    const size_t fstride,
    const peck_fft_cfg st,
    const size_t m) {

    peck_fft_cpx scratch[4];
    peck_fft_cpx * __restrict tw1, * __restrict tw2, * __restrict tw3;
    size_t k = m;
    const size_t m2 = 2*m;
    const size_t m3 = 3*m;

// printf("kf_bfly4, %d\n", fstride);

    tw3 = tw2 = tw1 = st->twiddles;

    if (st->inverse) {
        do {
            C_MUL(scratch[0], Fout[m], *tw1);
            C_MUL(scratch[3], Fout[m2], *tw2);
            C_MUL(scratch[2], Fout[m3], *tw3);

            C_SUB(scratch[1], *Fout, scratch[3]);
            C_ADDTO(*Fout, scratch[3]);
            
            C_ADD(scratch[3], scratch[0], scratch[2]);
            C_SUB(Fout[m2], *Fout, scratch[3]);
            C_ADDTO(*Fout, scratch[3]);

            tw1 += fstride;
            tw2 += fstride*2;
            tw3 += fstride*3;

            C_SUB(scratch[3], scratch[0], scratch[2]);
            Fout[m].r = scratch[1].r - scratch[3].i;
            Fout[m].i = scratch[1].i + scratch[3].r;
            Fout[m3].r = scratch[1].r + scratch[3].i;
            Fout[m3].i = scratch[1].i - scratch[3].r;

            ++Fout;
        } while (--k);
    }
    else {
        do {
            C_MUL(scratch[0], Fout[m], *tw1);
            C_MUL(scratch[3], Fout[m2], *tw2);
            C_MUL(scratch[2], Fout[m3], *tw3);

            C_SUB(scratch[1], *Fout, scratch[3]);
            C_ADDTO(*Fout, scratch[3]);
            
            C_ADD(scratch[3], scratch[0], scratch[2]);
            C_SUB(Fout[m2], *Fout, scratch[3]);
            C_ADDTO(*Fout, scratch[3]);

            tw1 += fstride;
            tw2 += fstride*2;
            tw3 += fstride*3;

            C_SUB(scratch[3], scratch[0], scratch[2]);
            Fout[m].r = scratch[1].r + scratch[3].i;
            Fout[m].i = scratch[1].i - scratch[3].r;
            Fout[m3].r = scratch[1].r - scratch[3].i;
            Fout[m3].i = scratch[1].i + scratch[3].r;

            ++Fout;
        } while (--k);
    }
}
#endif

static void kf_bfly3(
     peck_fft_cpx * Fout,
     const size_t fstride,
     const peck_fft_cfg st,
     size_t m) {

     size_t k=m;
     const size_t m2 = 2*m;
     peck_fft_cpx *tw1, *tw2;
     peck_fft_cpx scratch[5];
     peck_fft_cpx epi3;
     epi3 = st->twiddles[fstride*m];

printf("kf_bfly3\n");

     tw1 = tw2 = st->twiddles;

     do {
         C_MUL(scratch[1],Fout[m] , *tw1);
         C_MUL(scratch[2],Fout[m2] , *tw2);

         C_ADD(scratch[3],scratch[1],scratch[2]);
         C_SUB(scratch[0],scratch[1],scratch[2]);
         tw1 += fstride;
         tw2 += fstride*2;

         Fout[m].r = Fout->r - HALF_OF(scratch[3].r);
         Fout[m].i = Fout->i - HALF_OF(scratch[3].i);

         C_MULBYSCALAR( scratch[0] , epi3.i );

         C_ADDTO(*Fout,scratch[3]);

         Fout[m2].r = Fout[m].r + scratch[0].i;
         Fout[m2].i = Fout[m].i - scratch[0].r;

         Fout[m].r -= scratch[0].i;
         Fout[m].i += scratch[0].r;

         ++Fout;
     } while (--k);
}

static void kf_bfly5(
    peck_fft_cpx * Fout,
    const size_t fstride,
    const peck_fft_cfg st,
    int m
    ) {
    peck_fft_cpx *Fout0,*Fout1,*Fout2,*Fout3,*Fout4;
    int u;
    peck_fft_cpx scratch[13];
    peck_fft_cpx * twiddles = st->twiddles;
    peck_fft_cpx *tw;
    peck_fft_cpx ya,yb;
    ya = twiddles[fstride*m];
    yb = twiddles[fstride*2*m];

printf("kf_bfly5\n");

    Fout0 = Fout;
    Fout1 = Fout0 + m;
    Fout2 = Fout0 + 2*m;
    Fout3 = Fout0 + 3*m;
    Fout4 = Fout0 + 4*m;

    tw=st->twiddles;
    for (u = 0; u < m; ++u) {
        scratch[0] = *Fout0;

        C_MUL(scratch[1], *Fout1, tw[u*fstride]);
        C_MUL(scratch[2], *Fout2, tw[2*u*fstride]);
        C_MUL(scratch[3], *Fout3, tw[3*u*fstride]);
        C_MUL(scratch[4], *Fout4, tw[4*u*fstride]);

        C_ADD(scratch[7], scratch[1], scratch[4]);
        C_SUB(scratch[10], scratch[1], scratch[4]);
        C_ADD(scratch[8], scratch[2], scratch[3]);
        C_SUB(scratch[9], scratch[2], scratch[3]);

        Fout0->r += scratch[7].r + scratch[8].r;
        Fout0->i += scratch[7].i + scratch[8].i;

        scratch[5].r = scratch[0].r + S_MUL(scratch[7].r,ya.r) + S_MUL(scratch[8].r,yb.r);
        scratch[5].i = scratch[0].i + S_MUL(scratch[7].i,ya.r) + S_MUL(scratch[8].i,yb.r);

        scratch[6].r =  S_MUL(scratch[10].i,ya.i) + S_MUL(scratch[9].i,yb.i);
        scratch[6].i = -S_MUL(scratch[10].r,ya.i) - S_MUL(scratch[9].r,yb.i);

        C_SUB(*Fout1,scratch[5],scratch[6]);
        C_ADD(*Fout4,scratch[5],scratch[6]);

        scratch[11].r = scratch[0].r + S_MUL(scratch[7].r,yb.r) + S_MUL(scratch[8].r,ya.r);
        scratch[11].i = scratch[0].i + S_MUL(scratch[7].i,yb.r) + S_MUL(scratch[8].i,ya.r);
        scratch[12].r = - S_MUL(scratch[10].i,yb.i) + S_MUL(scratch[9].i,ya.i);
        scratch[12].i = S_MUL(scratch[10].r,yb.i) - S_MUL(scratch[9].r,ya.i);

        C_ADD(*Fout2,scratch[11],scratch[12]);
        C_SUB(*Fout3,scratch[11],scratch[12]);

        ++Fout0;++Fout1;++Fout2;++Fout3;++Fout4;
    }
}

/* perform the butterfly for one stage of a mixed radix FFT */
static void kf_bfly_generic(
    peck_fft_cpx * Fout,
    const size_t fstride,
    const peck_fft_cfg st,
    int m,
    int p) {
    
    int u,k,q1,q;
    peck_fft_cpx * twiddles = st->twiddles;
    peck_fft_cpx t;
    int Norig = st->nfft;

printf("kf_bfly_generic\n");

    peck_fft_cpx * scratch = (peck_fft_cpx*)PECK_FFT_TMP_ALLOC(sizeof(peck_fft_cpx)*p);

    for ( u=0; u<m; ++u ) {
        k=u;
        for (q1 = 0; q1 < p; ++q1) {
            scratch[q1] = Fout[k];
            k += m;
        }

        k=u;
        for ( q1=0 ; q1<p ; ++q1 ) {
            int twidx=0;
            Fout[ k ] = scratch[0];
            for (q=1;q<p;++q ) {
                twidx += fstride * k;
                if (twidx>=Norig) twidx-=Norig;
                C_MUL(t,scratch[q] , twiddles[twidx] );
                C_ADDTO( Fout[ k ] ,t);
            }
            k += m;
        }
    }
    PECK_FFT_TMP_FREE(scratch);
}

static void kf_work(
    peck_fft_cpx * Fout,
    const peck_fft_cpx * f,
    const size_t fstride,
    int *factors,
    const peck_fft_cfg st) {
    
    peck_fft_cpx *Fout_beg = Fout;
    const int p = *factors++; /* the radix */
    const int m = *factors++; /* stage's FFT length / p */
    const peck_fft_cpx *Fout_end = Fout + p*m;

// printf("kf_work, %d\n", fstride);

    if (m == 1) {
        do {
            *Fout = *f;
            f += fstride;
        } while (++Fout != Fout_end);
    } else {
        do {
            // recursive call:
            // DFT of size m*p performed by doing
            // p instances of smaller DFTs of size m, 
            // each one takes a decimated version of the input
            kf_work(Fout, f, fstride*p, factors, st);
            f += fstride;
        } while ((Fout += m) != Fout_end);
    }

    Fout = Fout_beg;

    // recombine the p smaller DFTs 
    switch (p) {
        case 2: 
            kf_bfly2(Fout, fstride, st, m); 
            break;
        case 3: kf_bfly3(Fout, fstride, st, m); break; 
        case 4: 
        {
//static unsigned counter = 0;        
//            armv7_cycles_start();
//            unsigned int t1 = armv7_cycles_read();
//printf("%08x %d %d\n", Fout, fstride, m);
            kf_bfly4(Fout, fstride, st, m); 
//            unsigned int t2 = armv7_cycles_read();
//            armv7_cycles_stop();
//            counter++;
//            if (counter > 150 && counter < 160) printf("XX %d\n", t2-t1);
}            
            break;
        case 5: kf_bfly5(Fout, fstride, st, m); break; 
        default: kf_bfly_generic(Fout, fstride, st, m, p); break;
    }
}

/*  
 * facbuf is populated by p1, m1, p2, m2, ...
 *   where 
 *     p[i] * m[i] = m[i-1]
 *     m0 = n                  
 */
static void kf_factor(int n, int * facbuf) {
    int p = 4;
    float floor_sqrt;
    floor_sqrt = floorf(sqrtf(n));

    /* factor out powers of 4, powers of 2, then any remaining primes */
    do {
        while (n % p) {
            switch (p) {
                case 4: p = 2; break;
                case 2: p = 3; break;
                default: p += 2; break;
            }
            if (p > floor_sqrt)
                p = n;          /* no more factors, skip to end */
        }
        n /= p;
        *facbuf++ = p;
        *facbuf++ = n;
    } while (n > 1);
}

/*
 * User-callable function to allocate all necessary storage space for the fft.
 * The return value is a contiguous block of memory, allocated with malloc.  As such,
 * it can be freed with free(), rather than a peck_fft-specific function.
 */
peck_fft_cfg peck_fft_alloc(int nfft, int inverse_fft, void *mem, size_t *lenmem) {
    peck_fft_cfg st = NULL;
    size_t memneeded = sizeof(struct peck_fft_state)
        + sizeof(peck_fft_cpx)*(nfft-1); /* twiddle factors */

    if (lenmem == NULL) {
        st = (peck_fft_cfg) PECK_FFT_MALLOC(memneeded);
    } else {
        if (mem != NULL && *lenmem >= memneeded)
            st = (peck_fft_cfg) mem;
        *lenmem = memneeded;
    }
    
    if (st) {
        int i;
        st->nfft=nfft;
        st->inverse = inverse_fft;

        for (i = 0; i < nfft; ++i) {
            const float pi = 3.14159265359f;
            float phase = -2*pi*i / nfft;
            if (st->inverse)
                phase *= -1;
            kf_cexp(st->twiddles+i, phase);
        }

        kf_factor(nfft, st->factors);
    }
    return st;
}

void peck_fft(peck_fft_cfg cfg, const peck_fft_cpx *fin, peck_fft_cpx *fout) {
    kf_work(fout, fin, 1, cfg->factors, cfg);
}

void peck_fft_cleanup(void) {
    /* nothing needed any more */
}

int peck_fft_next_fast_size(int n) {
    while (1) {
        int m = n;
        while ((m % 2) == 0) m /= 2;
        while ((m % 3) == 0) m /= 3;
        while ((m % 5) == 0) m /= 5;
        if (m <= 1)
            break; /* n is completely factorable by twos, threes, and fives */
        n++;
    }
    return n;
}
